--[[
    Attch Implements Manually Settings

    Adds the settings for the Attach Implements Manually mod to the general settings menu.

        @author:    BayernGamers
        @date:      07.06.2025
        @version:   1.0

        History:    v1.0 @07.06.2025 - initial implementation in FS25
                    -------------------------------------------------------------------------------------------

        License:    Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage:
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "AttachImplementsManuallySettings.lua")

AttachImplementsManuallySettings = {}
AttachImplementsManuallySettings.MOD_DIRECTORY = g_currentModDirectory
AttachImplementsManuallySettings.MOD_NAME = g_currentModName
AttachImplementsManuallySettings.MOD_SETTINGS_DIR = g_currentModSettingsDirectory

local AttachImplementsManuallySettings_mt = Class(AttachImplementsManuallySettings)

function AttachImplementsManuallySettings.new(modSettingsManager)
    if modSettingsManager == nil then
        log:printError("AttachImplementsManuallySettingsManager requires a modSettingsManager instance.")
        printCallstack()
        return nil
    end

    local self = setmetatable({}, AttachImplementsManuallySettings_mt)

    self.modSettingsManager = modSettingsManager

    return self
end

function AttachImplementsManuallySettings:init()
    local enabled, enableAutoAttachFrontloaderTools = self.modSettingsManager:getUserSettings()
    local settingsLayout = g_inGameMenu.generalSettingsLayout

    self:addTitle(g_i18n:getText("title_atttachImplementsManually"))

    local enabledState = enabled and 2 or 1
    local enableAutoAttachFrontloaderToolsState = enableAutoAttachFrontloaderTools and 2 or 1
    self:addCheckBox("enableAttachImplementsManually", g_i18n:getText("setting_enableAttachImplementsManually"), g_i18n:getText("toolTip_enableAttachImplementsManually"), enabledState, "onEnableAttachImplementsManuallyChanged")
    self:addCheckBox("enableAutoAttachFrontloaderTools", g_i18n:getText("setting_enableAutoAttachFrontloaderTools"), g_i18n:getText("toolTip_enableAutoAttachFrontloaderTools"), enableAutoAttachFrontloaderToolsState, "onEnableAutoAttachFrontloaderToolsChanged")

    settingsLayout:invalidateLayout()
end

function AttachImplementsManuallySettings:onEnableAttachImplementsManuallyChanged()
    local enabled, _ = self.modSettingsManager:getUserSettings()
    log:printDevInfo("Setting enableAttachImplementsManually's enabled to " .. tostring(not enabled), LoggingUtil.DEBUG_LEVELS.HIGH)
    self.modSettingsManager:setEnableAttachImplementsManually(not enabled)
end

function AttachImplementsManuallySettings:onEnableAutoAttachFrontloaderToolsChanged()
    local _, enableAutoAttachFrontloaderTools = self.modSettingsManager:getUserSettings()
    log:printDevInfo("Setting enableAutoAttachFrontloaderTools's enabled to " .. tostring(not enableAutoAttachFrontloaderTools), LoggingUtil.DEBUG_LEVELS.HIGH)
    self.modSettingsManager:setEnableAutoAttachFrontloaderTools(not enableAutoAttachFrontloaderTools)
end

function AttachImplementsManuallySettings:updateElementFocusIds(element)
    if not element then
        return
    end
    element.focusId = FocusManager:serveAutoFocusId()
    for _, child in pairs(element.elements) do
        self:updateElementFocusIds(child)
    end
end

function AttachImplementsManuallySettings:addTitle(text)
    local settingsLayout = g_inGameMenu.generalSettingsLayout
    local controlsList = g_inGameMenu.controlsList

    local title = nil
    for _, element in ipairs(settingsLayout.elements) do
        if element.name == "sectionHeader" then
            title = element:clone(settingsLayout)
            title:setText(text)

            self:updateElementFocusIds(title)
            table.insert(controlsList, title)
            return
        end
    end
end

function AttachImplementsManuallySettings:addCheckBox(id, text, toolTip, state, callback)
    local settingsLayout = g_inGameMenu.generalSettingsLayout
    local checkBoxTemplate = g_inGameMenu.checkWoodHarvesterAutoCutBox
    local controlsList = g_inGameMenu.controlsList

    local checkBox = checkBoxTemplate:clone(settingsLayout)
    checkBox.id = id

    local checkBoxElement = checkBox.elements[1]
    checkBoxElement.id = id
    checkBoxElement.target = self

    checkBoxElement:setCallback("onClickCallback", callback)
    checkBoxElement:setDisabled(false)

    local toolTipElement = checkBoxElement.elements[1]
    toolTipElement:setText(toolTip)

    local settingElement = checkBox.elements[2]
    settingElement:setText(text)

    checkBoxElement:setTexts({g_i18n:getText("ui_no"), g_i18n:getText("ui_yes")})
    checkBoxElement:setState(state)

    self:updateElementFocusIds(checkBox)
    table.insert(controlsList, checkBox)
end